﻿using System;
using System.Collections.Generic;
using System.IO.Ports;
using System.Management;
using System.Threading;
using static Spectral1.DATA_ACCESS.DA_Spectral;
using static Spectral1_VBClassLibrary.DataSet_Spectral;
using static Spectral1.BUSINESS_LOGIC.BL_KeyScaling;
using static Spectral1.BUSINESS_LOGIC.BL_Spectral;
using System.Diagnostics;
using System.ComponentModel;
using Spectral1_VBClassLibrary;
using System.Collections.Concurrent;
using System.Windows.Forms;

namespace Spectral1.DATA_ACCESS
{
    public class DA_USB_Comms
    {
        #region "================ DECLARATIONS ================"
        static SerialPort _serialPort = new SerialPort();
        USBDeviceInfo UDI = null;
        CodeGen_DS_Spectral _CGS;
        bool _suppress_comms = false;
        const byte message_start_char = 0x7E;
        const int progress_inc_value = 1;
        BackgroundWorker backgroundWorker1;
        //static readonly object QLock = new object();// Used to lock the queue by the two threads. See here :https://jonskeet.uk/csharp/threads/printable.html

        public bool flag_in_play_mode = true;

        Int32 di = 0;
        byte[] data = new byte[5000];
        int progress_value = 0;

        private class spectral_message
        {
            public Int32 byte_count;
            public byte[] buffer;
        }

        ConcurrentQueue<spectral_message> message_queue = new ConcurrentQueue<spectral_message>();

        #endregion

        #region "================ EVENTS ================================="
        private void backgroundWorker1_DoWork(object sender, System.ComponentModel.DoWorkEventArgs e)
        {
            while (true)
            {
                while (!message_queue.IsEmpty)
                {
                    spectral_message m;
                    message_queue.TryDequeue(out m);

                    string error_text = "";

                    if ((m != null) && (UDI != null) && (_suppress_comms == false))
                    {
                        try
                        {
                            _serialPort.Open();
                            Int32 payload_length = m.byte_count - 3;
                            byte payload_length_msb = (byte)(payload_length >> 8);
                            byte payload_length_lsb = (byte)(payload_length - (payload_length_msb << 8));

                            m.buffer[0] = message_start_char;
                            m.buffer[1] = payload_length_lsb;
                            m.buffer[2] = payload_length_msb;

                            _serialPort.Write(m.buffer, 0, m.byte_count);//(char)126 + "Z" + (char)126);

                        }
                        catch (Exception ex)
                        {
                            error_text = "Comms error : " + ex.ToString();
                            message_queue = new ConcurrentQueue<spectral_message>();//ConcurrentQueue doesn't have a clear method
                        }
                        finally
                        {
                            _serialPort.Close(); 
                            
                            //Hans passant warns of needing time to let a worker thread exit after the Dispose() or Close() call. https://stackoverflow.com/questions/7219653/why-is-access-to-com-port-denied
                            while (_serialPort.IsOpen == true) { }
                                                    
                            
                            Thread.Sleep(2); //Pause between items sent.  This is definitely needed otherwise can get tone processors 'dropping out' (crashing because they haven't had time to process).
                        }
                    }

                        progress_value += progress_inc_value;
                    if (progress_value > 100) { progress_value = progress_value - 100; }
                }
               

                Thread.Sleep(50); // Quarter of a second.ESSENTIAL in order to stop high CPU/Power usage in the app ! 

                ////This is ESSENTIAL in order to stop high CPU/Power usage in the app ! 
                //while (message_queue.Count == 0)
                //{
                //    Thread.Sleep(250); // Quarter of a second.
                //}
            }
        }

        #endregion


        #region "================ METHODS - USB LOW LEVEL ================"

        private void WaitForModuleToProcess()
        {
            if (IsConnected() == true)
                {
                    System.Threading.Thread.Sleep(module_pause_to_process_ms);
                }

        }
        public bool IsConnected()
        {
            return (UDI != null);
        }

        public void SendUSBMessage(Int32 byte_count, byte[] buffer)
        {
            spectral_message m = new spectral_message();
            m.buffer = new byte[byte_count];
            for (int i = 0; i < byte_count; i++) { m.buffer[i] = buffer[i]; }
            m.byte_count = byte_count;
            message_queue.Enqueue(m);

        }

        public DA_USB_Comms(CodeGen_DS_Spectral CGS)
        {
            _CGS = CGS;
            backgroundWorker1 = new BackgroundWorker();
            backgroundWorker1.WorkerSupportsCancellation = false;
            backgroundWorker1.WorkerReportsProgress = false;
            backgroundWorker1.DoWork += backgroundWorker1_DoWork;
            backgroundWorker1.RunWorkerAsync();
        }

        public bool suppress_comms
        {
            get { return this._suppress_comms; }
            set { _suppress_comms = value; }
        }

        public void Disconnect()
        {
            if (_serialPort.IsOpen == true) { _serialPort.Close(); }
            UDI = null;
        }

        public string InitialiseSerialPort(USBDeviceInfo UDI)
        {
            string error_text = "";

            if (UDI != null)
            {
                _serialPort.PortName = UDI.DeviceID; //The Name is something like "USB Serial Device (COM4)" but the Device ID is "COM4"
                _serialPort.BaudRate = 9600;
                _serialPort.Parity = Parity.None;
                _serialPort.DataBits = 8;
                _serialPort.StopBits = StopBits.One;

                //Set the read / write timeouts
                _serialPort.ReadTimeout = 500;
                _serialPort.WriteTimeout = 2000;
                _serialPort.WriteBufferSize = 4096; //The WriteBufferSize property ignores any value smaller than 2048

                //Cler the message queue
                message_queue = new ConcurrentQueue<spectral_message>();//ConcurrentQueue doesn't have a clear method

                //Wait a little for the serial port to be ready (don't know if this is necessary)
                Thread.Sleep(5);

                //Send some bytes to 'flush' the port. Have found that unless this is done then can initially get confused.
                //The bytes ca be any value except the message_start_byte.
                try
                {
                    byte[] buffer = new byte[10];
                    _serialPort.Open();
                    for (int i = 0; i < 10; i++)
                    { buffer[i] = 210; }
                    _serialPort.Write(buffer, 0, 10);
                    _serialPort.Close(); 
                    Thread.Sleep(5);
                }
                catch (Exception ex)
                {
                    error_text = "Comms error opening serial port : " + ex.ToString();
                    message_queue = new ConcurrentQueue<spectral_message>();//ConcurrentQueue doesn't have a clear method
                }
            }

            return error_text;
        }

        public USBDeviceInfo FindSpectralSMDevice()
        {
            ManagementObjectCollection collection;
            using (var searcher = new ManagementObjectSearcher(@"Select * From Win32_SerialPort"))
            {
                collection = searcher.Get();
                if (collection != null)
                {
                    if (collection.Count > 0)
                    {
                        foreach (ManagementObject mo in collection)
                        {
                            UDI = new USBDeviceInfo(
                                (String)mo.GetPropertyValue("DeviceID"),
                                (String)mo.GetPropertyValue("PNPDeviceID"),
                                (String)mo.GetPropertyValue("Description"),
                                (String)mo.GetPropertyValue("Name")
                                );
                            if (UDI.PnpDeviceID.Contains("VID_04D8&PID_ED20"))
                            {
                                return UDI;
                            }
                        }
                    }
                }
            }
            return null;
        }
        #endregion

        #region "================= METHODS - MESSAGE COMPILATION - HELPERS ================="
        private void msg_compile_recipient(msg_recipients r)
        {
            di++;
            data[di] = (byte)r;
        }

        private void msg_compile_message_type(msg_types t)
        {
            di++;
            data[di] = (byte)t;
        }

        private void msg_compile_byte(byte b)
        {
            di++;
            data[di] = b;
        }

        private void msg_compile_uint16(UInt16 i)
        {
            if ((i == (int)(msg_types.StopPlayModeTP)) || (i == (int)(msg_types.StartPlayModeTP))) { i = (int)(msg_types.StartStopPlayModeTP_Replacement); }
            byte MSB = (byte)(i >> 8);
            byte LSB = (byte)(i - (MSB << 8));
            di++;
            data[di] = LSB;
            di++;
            data[di] = MSB;
        }

        #endregion

        #region "================= METHODS - MESSAGE COMPILATION ================="
        public bool msg_compile_adsr_section(Int32 patch_id, Int32 adsr_section_id)
        {
            adsr_sectionRow r = _CGS.Table_adsr_section.GetRow(patch_id, adsr_section_id);
            if (r != null)
            {
                msg_compile_byte((Byte)adsr_section_id);
                msg_compile_uint16((UInt16)r.flag_active);
                msg_compile_uint16((UInt16)r.end_time_ms);
                msg_compile_uint16((UInt16)get_ks_swx32_for_value_usw((UInt16)r.end_time_ms, (Double)r.end_time_ms_keyscale_upper));
                msg_compile_uint16((UInt16)get_ks_swx32_for_value_usw((UInt16)r.end_time_ms, (Double)r.end_time_ms_keyscale_lower));
                msg_compile_uint16((UInt16)r.inharmonic_sample_playback_mode_id);
                msg_compile_uint16((UInt16)r.inharmonic_sample_id);
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_adsr_section_envelope_config(Int32 patch_id, Int32 adsr_section_id, Int32 adsr_section_envelope_config_id)
        {
            adsr_section_envelope_configRow r = _CGS.Table_adsr_section_envelope_config.GetRow(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            if (r != null)
            {
                msg_compile_byte((Byte)adsr_section_id);
                msg_compile_byte((Byte)adsr_section_envelope_config_id);
                msg_compile_uint16((UInt16)r.depth_env_type_id);
                if ((adsr_section_envelope_config_id == Convert.ToInt32(envelope.env_vibrato)) || (adsr_section_envelope_config_id == Convert.ToInt32(envelope.env_timbre_lfo)))
                {
                    msg_compile_uint16(LogFnPercTo32766(r.depth_env_target));
                }
                else
                {
                    msg_compile_uint16((UInt16)(r.depth_env_target * 655.34));
                }
                msg_compile_uint16((UInt16)get_ks_sw_for_value_usp0to100((UInt16)r.depth_env_target, (Double)r.depth_env_target_keyscale_upper));
                msg_compile_uint16((UInt16)get_ks_sw_for_value_usp0to100((UInt16)r.depth_env_target, (Double)r.depth_env_target_keyscale_lower));

                if (adsr_section_envelope_config_id == Convert.ToInt32(envelope.env_amplitude))
                {
                    msg_compile_uint16((UInt16)(r.depth_env_lin_delta));
                    msg_compile_uint16((UInt16)(r.depth_env_exp_multiplier * 655.35));
                }
                else
                {
                    //Errr ...this is the same ???
                    msg_compile_uint16((UInt16)r.depth_env_lin_delta);
                    msg_compile_uint16((UInt16)(r.depth_env_exp_multiplier * 655.35));
                }

                msg_compile_uint16((UInt16)get_ks_swx32_for_value_usw((UInt16)r.depth_env_lin_delta, (Double)r.depth_env_lin_delta_keyscale_upper));
                msg_compile_uint16((UInt16)get_ks_swx32_for_value_usw((UInt16)r.depth_env_lin_delta, (Double)r.depth_env_lin_delta_keyscale_lower));

                msg_compile_uint16((UInt16)get_ks_swx32_for_value_sp0to100((Double)r.depth_env_exp_multiplier, (Double)r.depth_env_exp_multiplier_keyscale_upper));
                msg_compile_uint16((UInt16)get_ks_swx32_for_value_sp0to100((Double)r.depth_env_exp_multiplier, (Double)r.depth_env_exp_multiplier_keyscale_lower));
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_biquad_eq_calculated(Int32 patch_id, Int32 biquad_eq_calculated_id)
        {
            biquad_eq_calculatedRow r = _CGS.Table_biquad_eq_calculated.GetRow(patch_id, biquad_eq_calculated_id);
            if (r != null)
            {
                msg_compile_byte((Byte)biquad_eq_calculated_id);
                msg_compile_uint16((UInt16)(r.a0 * 32766));
                msg_compile_uint16((UInt16)(r.a1 * 32766));
                msg_compile_uint16((UInt16)(r.a2 * 32766));
                msg_compile_uint16((UInt16)(r.b1 * 32766));
                msg_compile_uint16((UInt16)(r.b2 * 32766));
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_lfo_envelope_config(Int32 patch_id, Int32 lfo_envelope_config_id)
        {
            lfo_envelope_configRow r = _CGS.Table_lfo_envelope_config.GetRow(patch_id, lfo_envelope_config_id);
            if (r != null)
            {
                msg_compile_byte((Byte)lfo_envelope_config_id);
                //NOTE: Can't currently explain why need the *2 in this below in order to get true freq ...!!!
                if (lfo_envelope_config_id == (Int32)lfos.lfo_tremolo)
                {
                    //Tremolo is updated every 1ms on the Tone Processor, otherwise the steps are heard with high freq components
                    msg_compile_uint16((UInt16)((double)r.default_wt_inc_q11_5 * ((double)(32 * 2 * samples_per_ms)) / (double)A0FrequencyHz));
                }
                else
                {
                    //Non- tremolo are all updated at 10ms interval
                    msg_compile_uint16((UInt16)((double)r.default_wt_inc_q11_5 * ((double)(32 * 2 * samples_per_ms * lfo_event_period)) / (double)A0FrequencyHz));
                }

                msg_compile_uint16((UInt16)r.depth_cc_source);
                msg_compile_uint16((UInt16)r.enabled);
                msg_compile_uint16((UInt16)r.lfo_wave_type_id);
                msg_compile_uint16((UInt16)r.freq_cc_source);
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_save_patch_to_module(Int32 module_patch_id)
        {
            msg_compile_byte((Byte)module_patch_id);
            return true;
        }

        private Int32 ConvertPatchIDToModuleID(Int32 patch_set_id, Int32 p_id)
        {
            Int32 module_id = 0;
            for (Int32 patch_set_patches_id = 0; patch_set_patches_id < patches_per_patchset; patch_set_patches_id++)
            {
                Int32 patch_id = _CGS.Table_patch_set_patches.GetRow(patch_set_id, patch_set_patches_id).patch_id;
                if (patch_id == p_id)
                { return patch_set_patches_id; }
            }
            return module_id;
        }

        public bool msg_compile_send_to_module_msg_default_banks()
        {
            system_settingsRow s = _CGS.Table_system_settings.GetRow(0);
            if (s == null) { return false; }
            int default_performance_bank = s.default_performance_bank;
            int default_patch_bank = s.default_patch_bank;
            int lsb;
            int msb;
            msb = default_performance_bank >> 7; //Note >>7 not 8
            lsb = default_performance_bank - (msb << 7);
            msg_compile_byte((Byte)lsb);
            msg_compile_byte((Byte)msb);
            msb = default_patch_bank >> 7; //Note >>7 not 8
            lsb = default_patch_bank - (msb << 7);
            msg_compile_byte((Byte)lsb);
            msg_compile_byte((Byte)msb);
            return true;
        }

        public bool msg_compile_send_to_module_performance(Int32 patchset_id,Int32 performance_id)
        {
            patch_setRow psr = _CGS.Table_patch_set.GetRow(patchset_id);
            performanceRow pr = _CGS.Table_performance.GetRow(patchset_id, performance_id);
            if ((psr == null)||(pr == null)) { return false; }

            //performance id
            msg_compile_byte((Byte)performance_id);

            //Performance channels
            for (Int32 pcr_id = 0; pcr_id < max_midi_channels; pcr_id++)
            {
                performance_channelRow pcr = _CGS.Table_performance_channel.GetRow(patchset_id, performance_id, pcr_id);
                if (pcr == null) { return false; }
                msg_compile_byte((Byte)ConvertPatchIDToModuleID(patchset_id,pcr.patch_id));
                msg_compile_byte((Byte)(pcr.volume * 2.55));
                msg_compile_byte((Byte)pcr.note_range_low);
                msg_compile_byte((Byte)pcr.note_range_high);
            }

            //HAAS delay for the performance
                
            msg_compile_byte((Byte)pr.haas_delay_ms);

            //Channels allocated to processors
            for (Int32 pp_id = 0; pp_id < max_tone_processors; pp_id++)
            {
                performance_processorRow ppr = _CGS.Table_performance_processor.GetRow(patchset_id, performance_id, pp_id);
                if (ppr == null) { return false; }
                msg_compile_byte((Byte)ppr.channel_id);
            }
            return true;
        }

        public bool msg_compile_envelope_control(Int32 patch_id, Int32 envelope_control_id)
        {
            envelope_controlRow r = _CGS.Table_envelope_control.GetRow(patch_id, envelope_control_id);
            if (r != null)
            {
                msg_compile_byte((Byte)envelope_control_id);
            
                msg_compile_uint16(Convert.ToUInt16(r.depth_initial_level * 655.35));    //OLD 311221: NOTE 655.34. I don't know why 655.35 doesn't work !!!!
                msg_compile_uint16((UInt16)r.gain_cc_source);
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_patch_misc(Int32 patch_id)
        {
            patchRow r = _CGS.Table_patch.GetRow(patch_id);
            if (r != null)
            {
                msg_compile_uint16((UInt16)r.active_layers);
                msg_compile_uint16((UInt16)(r.degree_of_random_detuning * 655.35));
                msg_compile_uint16((UInt16)(r.degree_of_random_phase * 655.35));
                msg_compile_uint16((UInt16)(r.degree_of_regular_detuning * 655.35));
                msg_compile_uint16((UInt16)r.pitch_bend_enabled);
                msg_compile_uint16((UInt16)r.portamento_enabled);
                msg_compile_uint16((UInt16)r.portamento_rate);
                msg_compile_uint16((UInt16)r.sustain_enabled);
                msg_compile_uint16((UInt16)r.detuning_mode_id);
                //msg_compile_uint16((UInt16)r.patch_category_id);
                if (r.detuning_mode_id == Convert.ToInt32(detune_modes.dm_cents))
                {
                    msg_compile_uint16((UInt16)(r.layer0_st_inc_q11_5_detune * 655.34));
                    msg_compile_uint16((UInt16)(r.layer1_st_inc_q11_5_detune * 655.34));
                }
                else
                {
                    msg_compile_uint16((UInt16)(r.layer0_st_inc_q11_5_detune * detune_linear_value_per_hz));
                    msg_compile_uint16((UInt16)(r.layer1_st_inc_q11_5_detune * detune_linear_value_per_hz));
                }
                msg_compile_uint16((UInt16)(r.layer2_st_inc_q11_5_detune * 655.35));//THIS IS REALLY PATCH GAIN !! THIS IS UNUSED IN THE MODULE BECAUSE APPIED IN THE APP ON THE HARMONIC LEVELS WHEN SENDING (need to rename field in the Database)

                msg_compile_uint16((UInt16)r.key_scale_split_note_id);
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_waveform_harmonics(Int32 waveform_set_id, Int32 note_sector_id, Int32 intensity_layer_id, Int32 waveform_id, Double amplitude_scaling)
        {
            msg_compile_byte((byte)note_sector_id);
            msg_compile_byte((byte)intensity_layer_id);
            msg_compile_byte((byte)waveform_id);
            for (int h = 0; h < max_harmonics; h++)
            {
                waveform_harmonicRow r = _CGS.Table_waveform_harmonic.GetRow(waveform_set_id, note_sector_id, intensity_layer_id, waveform_id, h);
                if (r != null)
                {
                    int b = (int)((double)r.level * amplitude_scaling); 
                    if (b > 255) { b = 255; }
                    msg_compile_byte((byte)(b));
                }
                else
                {
                    return false;
                    //msg_compile_byte((byte)(0));
                }
            }
            return true;
        }

        public bool msg_compile_waveform_set(Int32 waveform_set_id)
        {
            waveform_setRow r = _CGS.Table_waveform_set.GetRow(waveform_set_id);
            if (r != null)
            {
                msg_compile_uint16((UInt16)r.intensity_controller_source);
                msg_compile_uint16((UInt16)r.waveform_controller_source);
                msg_compile_uint16((UInt16)r.timbre_mode);
                return true;
            }
            else
            { return false; }
        }

        public bool msg_compile_body_resonance_filter_band(Int32 body_resonance_filter_id, Int32 b, Double filter_scaling_factor)
        {
            UInt16 lu;
            body_resonance_filter_bandRow r;

            r = _CGS.Table_body_resonance_filter_band.GetRow(body_resonance_filter_id, b);
            msg_compile_byte((byte)b);
            if (r != null)
            {
                lu = (UInt16)((Double)r.level * filter_scaling_factor * 64);//64 added 180121 because of recent new filter design.
                msg_compile_uint16(lu);
                msg_compile_uint16((UInt16)Convert.ToInt16(r.slope));
                return true;
            }
            else
            {
                return false;
                //msg_compile_uint16(0);
                //msg_compile_uint16(0);
            }
        }

        public void RecalcFilterSlopes(Int32 body_resonance_filter_id, Double filter_scaling_factor)
        {
            Double accurate_slope;
            Int16 mcu_slope_mult32;
            
            for (Int32 b = 0; b < (max_body_resonance_filter_bands - 1); b++)
            {
                body_resonance_filter_bandRow brfbr = _CGS.Table_body_resonance_filter_band.GetRow(body_resonance_filter_id, b);
                if (brfbr == null)
                {
                    return;
                }

                Double this_band_level = Convert.ToInt32(brfbr.level * filter_scaling_factor);
                Double next_band_level = Convert.ToInt32(_CGS.Table_body_resonance_filter_band.GetRow(body_resonance_filter_id, b + 1).level * filter_scaling_factor);
                Double delta_level = next_band_level - this_band_level;

                Double this_band_freq = calc_band_freq(b);
                Double next_band_freq = calc_band_freq(b + 1);
                Double delta_freq = next_band_freq - this_band_freq;

                accurate_slope = delta_level / delta_freq;
                if (accurate_slope >= 1024)
                {
                    accurate_slope = 1023; //NOTE: Not 1024 because 1024 * 32 = 32768 = -1
                }
                else if (accurate_slope < -1024)
                {
                    accurate_slope = -1024;
                }

                mcu_slope_mult32 = (Int16)(accurate_slope * 32);
                brfbr.slope = mcu_slope_mult32;
            }
        }
        #endregion

        #region "================ METHODS - SENDING MESSAGES : HIGH-LEVEL ================="
        public void SendAllWaveformHarmonics(Int32 waveform_set_id, Int32 patch_id, double level_scaling)
        {
            if (waveform_set_id == 0) { return; }

            for (Int32 n = 0; n < max_note_sectors; n++)
            {
                for (Int32 l = 0; l < max_intensity_layers; l++)
                {
                    for (Int32 waveform_id = 0; waveform_id < max_waveforms; waveform_id++)
                    {
                        send_to_module_waveform_harmonics(waveform_set_id, n, l, waveform_id, patch_id, level_scaling);
                    }
                }
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_body_resonance_filter_bands(Int32 body_resonance_filter_id, double filter_scaling_factor)
        {
            RecalcFilterSlopes(body_resonance_filter_id, filter_scaling_factor);

            for (Int32 b = 0; b < max_body_resonance_filter_bands; b++)
            {
                send_to_module_body_resonance_filter_band(body_resonance_filter_id, b, filter_scaling_factor);
            }

            WaitForModuleToProcess();
        }


        public void send_to_module_total_patch_config(Int32 patch_id, double level_scaling, double filter_scaling_factor)
        {
            patchRow pr = _CGS.Table_patch.GetRow(patch_id);
            if (pr == null) { return; }

            Int32 waveform_set_id = pr.waveform_set_id;
            Int32 body_resonance_filter_id = pr.body_resonance_filter_id;

            send_to_module_patch_misc(patch_id);
            WaitForModuleToProcess();//added

            for (Int32 adsr_section_id = 0; adsr_section_id < max_adsr_sections; adsr_section_id++)
            {
                send_to_module_adsr_section(patch_id, adsr_section_id);
                for (Int32 adsr_section_envelope_config_id = 0; adsr_section_envelope_config_id < max_envelopes; adsr_section_envelope_config_id++)
                {
                    send_to_module_adsr_section_envelope_config(patch_id, adsr_section_id, adsr_section_envelope_config_id);
                }
                WaitForModuleToProcess();//added
            }

            for (Int32 lfo_envelope_config_id = 0; lfo_envelope_config_id < max_lfos; lfo_envelope_config_id++)
            {
                send_to_module_lfo_envelope_config(patch_id, lfo_envelope_config_id);
                WaitForModuleToProcess();//added
            }

            for (Int32 e = 0; e < max_envelopes; e++)
            {
                send_to_module_envelope_control_config(patch_id, e);
            }
            WaitForModuleToProcess();

            SendAllWaveformHarmonics(waveform_set_id, patch_id, level_scaling);
            WaitForModuleToProcess();

            send_to_module_waveform_set(waveform_set_id);
            WaitForModuleToProcess();

            send_to_module_body_resonance_filter_bands(body_resonance_filter_id, filter_scaling_factor);
            WaitForModuleToProcess();
        }

        #endregion

        #region "================ METHODS - SENDING MESSAGES : LOW-LEVEL =================="

        private void ce(bool compile_was_success)
        {
            //Checks for compile error
            if (compile_was_success == false) { MessageBox.Show("Compile error ! Communication with Module aborted.","Comms Error", MessageBoxButtons.OK,MessageBoxIcon.Error); }
        }

        public void send_to_module_start_play_mode()
        {
            if (flag_in_play_mode == false)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.StartPlayMode);
                msg_compile_byte(0);//There has to be a payload
                Debug.WriteLine("*** send_to_module_start_play_mode ***");
                SendUSBMessage(di + 1, data);
                flag_in_play_mode = true;
                WaitForModuleToProcess();
            }
        }

        public void send_to_module_stop_play_mode()
        {
            if (flag_in_play_mode == true)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.StopPlayMode);
                msg_compile_byte(0);//There has to be a payload
                Debug.WriteLine("*** send_to_module_stop_play_mode ***");
                SendUSBMessage(di + 1, data);
                flag_in_play_mode = false;
                WaitForModuleToProcess();
            }
        }

        public void send_to_module_adsr_section(Int32 patch_id, Int32 adsr_section_id)
        {
            adsr_sectionRow r = _CGS.Table_adsr_section.GetRow(patch_id, adsr_section_id);
            if (r != null)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.msg_type_adsr_section);
                ce(msg_compile_adsr_section(patch_id, adsr_section_id)); 
                Debug.WriteLine("send_to_module_adsr_section(" + patch_id.ToString() + "," + adsr_section_id.ToString() + ")");
                SendUSBMessage(di + 1, data);
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_adsr_section_envelope_config(Int32 patch_id, Int32 adsr_section_id, Int32 adsr_section_envelope_config_id)
        {
            adsr_section_envelope_configRow r = _CGS.Table_adsr_section_envelope_config.GetRow(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            if (r != null)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.msg_type_adsr_section_envelope_config);
                ce(msg_compile_adsr_section_envelope_config(patch_id, adsr_section_id, adsr_section_envelope_config_id));
                Debug.WriteLine("send_to_module_adsr_section_envelope_config(" + patch_id.ToString() + "," + adsr_section_id.ToString() + "," + adsr_section_envelope_config_id.ToString() + ")");
                SendUSBMessage(di + 1, data);
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_lfo_envelope_config(Int32 patch_id, Int32 lfo_envelope_config_id)
        {
            lfo_envelope_configRow r = _CGS.Table_lfo_envelope_config.GetRow(patch_id, lfo_envelope_config_id);
            if (r != null)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.msg_type_lfo_envelope_config);
                ce(msg_compile_lfo_envelope_config(patch_id, lfo_envelope_config_id));
                Debug.WriteLine("send_to_module_lfo_envelope_config(" + patch_id.ToString() + "," + lfo_envelope_config_id.ToString() + ")");
                SendUSBMessage(di + 1, data);
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_envelope_control_config(Int32 patch_id, Int32 envelope_control_id)
        {
            envelope_controlRow r = _CGS.Table_envelope_control.GetRow(patch_id, envelope_control_id);
            if (r != null)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.msg_type_envelope_control);
                ce(msg_compile_envelope_control(patch_id, envelope_control_id));
                Debug.WriteLine("send_to_module_envelope_control_config(" + patch_id.ToString() + "," + envelope_control_id.ToString() + ")");
                SendUSBMessage(di + 1, data);
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_save_patch_to_module(Int32 module_patch_id)
        {
            di = 2;
            msg_compile_recipient(msg_recipients.mixer);
            msg_compile_message_type(msg_types.msg_type_save_patch_to_EEPROM);
            ce(msg_compile_save_patch_to_module(module_patch_id));
            Debug.WriteLine("send_to_module_save_patch_to_module(" + module_patch_id.ToString() + ")");
            SendUSBMessage(di + 1, data);

            //Now wait until the module has written all the patch data to EEPROM
            System.Threading.Thread.Sleep(module_patch_write_to_EEPROM_duration_ms);
        }

        public void send_to_module_msg_default_banks()
        {
            di = 2;
            msg_compile_recipient(msg_recipients.master);
            msg_compile_message_type(msg_types.msg_default_banks);
            ce(msg_compile_send_to_module_msg_default_banks());
            Debug.WriteLine("send_to_module_msg_default_banks()");
            SendUSBMessage(di + 1, data);

            //Now wait so that the Master SPI buffer doesn't overload
            WaitForModuleToProcess();
        }

        public void send_to_module_performance(Int32 patch_set_id,Int32 performance_id)
        {
            di = 2;
            msg_compile_recipient(msg_recipients.master);
            msg_compile_message_type(msg_types.msg_type_performance);
            ce(msg_compile_send_to_module_performance(patch_set_id, performance_id));
            Debug.WriteLine("send_to_module_performances(" + patch_set_id.ToString() + ")");
            SendUSBMessage(di + 1, data);

            //Now wait so that the Master SPI buffer doesn't overload
            WaitForModuleToProcess();
        }

        public void send_to_module_relay_performances_to_mixer()
        {
            di = 2;
            msg_compile_recipient(msg_recipients.master);
            msg_compile_message_type(msg_types.msg_type_relay_performances_to_mixer);
            msg_compile_byte(0);//There has to be a payload
            Debug.WriteLine("send_to_module_relay_performances_to_mixer");
            SendUSBMessage(di + 1, data);
            WaitForModuleToProcess();
        }

        public void send_to_module_select_performance(Int32 performance_id)
        {
            di = 2;
            msg_compile_recipient(msg_recipients.master);
            msg_compile_message_type(msg_types.msg_type_select_performance);
            msg_compile_byte((byte)performance_id);
            Debug.WriteLine(" send_to_module_select_performance : " + performance_id.ToString());
            SendUSBMessage(di + 1, data);
            WaitForModuleToProcess();
        }

        public void send_to_module_patch_misc(Int32 patch_id)
        {
            patchRow r = _CGS.Table_patch.GetRow(patch_id);
            if (r != null)
            {
                di = 2;
                msg_compile_recipient(msg_recipients.mixer);
                msg_compile_message_type(msg_types.msg_type_patch);
               ce( msg_compile_patch_misc(patch_id));
                Debug.WriteLine("send_to_module_patch_misc(" + patch_id.ToString() + ")");
                SendUSBMessage(di + 1, data);
            }
            WaitForModuleToProcess();
        }

        public void send_to_module_waveform_harmonics(Int32 waveform_set_id, Int32 note_sector_id, Int32 intensity_layer_id, Int32 waveform_id, Int32 patch_id,double level_scaling)
        {
            patchRow r = _CGS.Table_patch.GetRow(patch_id);
            Double amplitude_scaling = (r.layer2_st_inc_q11_5_detune/100.0) * level_scaling;

            di = 2;
            msg_compile_recipient(msg_recipients.mixer);
            msg_compile_message_type(msg_types.msg_type_waveform_harmonic);
            ce(msg_compile_waveform_harmonics(waveform_set_id, note_sector_id, intensity_layer_id, waveform_id, amplitude_scaling));
            Debug.WriteLine("send_to_module_waveform_harmonics(" + waveform_set_id.ToString() + "," + note_sector_id.ToString() + "," + intensity_layer_id.ToString() + "," + waveform_id.ToString() + "," + amplitude_scaling.ToString() + ")");
            SendUSBMessage(di + 1, data);

            WaitForModuleToProcess();
        }

        public void send_to_module_waveform_set(Int32 waveform_set_id)
        {
            if (waveform_set_id == 0) { return; }

            di = 2;
            msg_compile_recipient(msg_recipients.mixer);
            msg_compile_message_type(msg_types.msg_type_waveform_set);
            ce( msg_compile_waveform_set(waveform_set_id));
            Debug.WriteLine("send_to_module_waveform_set(" + waveform_set_id.ToString() + ")");
            SendUSBMessage(di + 1, data);
            WaitForModuleToProcess();
        }

        public void send_to_module_body_resonance_filter_band(Int32 body_resonance_filter_id, Int32 b,double filter_scaling_factor)
        {
            di = 2;
            msg_compile_recipient(msg_recipients.mixer);
            msg_compile_message_type(msg_types.msg_type_body_resonance_filter_band);
            ce(msg_compile_body_resonance_filter_band(body_resonance_filter_id, b, filter_scaling_factor));
            Debug.WriteLine("send_to_module_body_resonance_filter_band(" + body_resonance_filter_id.ToString() + ":" + b.ToString() + ")");
            SendUSBMessage(di + 1, data);

            WaitForModuleToProcess();
        }

        #endregion

    }

    #region "================ SUB CLASSES ================"
    public class USBDeviceInfo
    {
        public USBDeviceInfo(string deviceID, string pnpDeviceID, string description, string Name)
        {
            this.DeviceID = deviceID;
            this.PnpDeviceID = pnpDeviceID;
            this.Description = description;
            this.Name = Name;
        }
        public string Name { get; private set; }
        public string DeviceID { get; private set; }
        public string PnpDeviceID { get; private set; }
        public string Description { get; private set; }
    }

    #endregion

}
